/**
  ******************************************************************************
  * @file    Demonstrations/CM7/Src/main.c
  * @author  MCD Application Team
  * @brief   This demo describes how display bmp images from SD card on LCD using
             the Adafruit 1.8" TFT shield with Joystick and microSD mounted on top
             of the STM32 Nucleo board.
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2018 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup STM32H7xx_HAL_Demonstrations
  * @{
  */

/** @addtogroup Demo
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
typedef enum
{
  SHIELD_NOT_DETECTED = 0,
  SHIELD_DETECTED
}ShieldStatus;

/* Private define ------------------------------------------------------------*/
#define SD_CARD_NOT_FORMATTED                    0
#define SD_CARD_FILE_NOT_SUPPORTED               1
#define SD_CARD_OPEN_FAIL                        2
#define FATFS_NOT_MOUNTED                        3
#define BSP_SD_INIT_FAILED                       4
#define SD_CARD_NO_FILE                          5

#define POSITION_X_BITMAP                        0
#define POSITION_Y_BITMAP                        0

#define FPU_ENABLED								 1
#define FP_DOUBLE								 0

#define HSEM_ID_0 (0U) /* HW semaphore 0*/
#define HSEM_ID_1 (1U) /* HW semaphore 1*/
#define HSEM_ID_2 (2U) /* HW semaphore 2*/

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint8_t BlinkSpeed = 0, str[20];
__IO uint8_t JoystickValue = 0;
char* pDirectoryFiles[MAX_BMP_FILES];
FATFS SD_FatFs;  /* File system object for SD card logical drive */
char SD_Path[4]; /* SD card logical drive path */

/* Private function prototypes -----------------------------------------------*/
static void MPU_Config(void);
static void SystemClock_Config(void);
static void CPU_CACHE_Enable(void);
static void Error_Handler(void);
//static void LED1_Blink(void);
static void LED3_Blink(void);
//static void LED1_On(void);
static ShieldStatus TFT_ShieldDetect(void);
static void SDCard_Config(void);
static void TFT_DisplayErrorMessage(uint8_t message);

static void TFT_Display_Header(void);
static void TFT_Display_Desc(uint8_t *desc);
static void TFT_Display_Menu(uint8_t *desc);
static void TFT_Display_result(uint8_t *desc, uint8_t *result, uint8_t *unit);
//static void TFT_Display_Progress(int current, uint32_t max);
static void TFT_Display_Progress_1(uint32_t n);

/* benchmark parameters */

#define SH_RESULT_ADDR 0x30000000
volatile uint32_t *result = SH_RESULT_ADDR;

/* Private functions ---------------------------------------------------------*/

/* I2C handler declaration */
//I2C_HandleTypeDef I2cHandle;

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
  int32_t timeout;
	int8_t n = 0;

  /* System Init, System clock, voltage scaling and L1-Cache configuration are done by CPU1 (Cortex-M7)
     in the meantime Domain D2 is put in STOP mode(Cortex-M4 in deep-sleep)
  */

  /* Configure the MPU attributes */
  MPU_Config();

  /* Enable the CPU Cache */
  CPU_CACHE_Enable();

  /* Wait until CPU2 boots and enters in stop mode or timeout*/
  timeout = 0xFFFF;
  while((__HAL_RCC_GET_FLAG(RCC_FLAG_D2CKRDY) != RESET) && (timeout-- > 0));
  if ( timeout < 0 )
  {
    Error_Handler();
  }

  /* STM32H7xx HAL library initialization:
       - Systick timer is configured by default as source of time base, but user
         can eventually implement his proper time base source (a general purpose
         timer for example or other time source), keeping in mind that Time base
         duration should be kept 1ms since PPP_TIMEOUT_VALUEs are defined and
         handled in milliseconds basis.
       - Set NVIC Group Priority to 4
       - Low Level Initialization
     */
  HAL_Init();

  /* Configure the system clock to 400 MHz */
  SystemClock_Config();

  /* When system initialization is finished, Cortex-M7 will release Cortex-M4  by means of
     HSEM notification */

  /*HW semaphore Clock enable*/
  __HAL_RCC_HSEM_CLK_ENABLE();

  /*Take HSEM */
  HAL_HSEM_FastTake(HSEM_ID_0);
  /*Release HSEM in order to notify the CPU2(CM4)*/
  HAL_HSEM_Release(HSEM_ID_0,0);

  /* Check the availability of adafruit 1.8" TFT shield on top of STM32NUCLEO
     board. This is done by reading the state of IO PF.03 pin (mapped to JoyStick
     available on adafruit 1.8" TFT shield). If the state of PF.03 is high then
     the adafruit 1.8" TFT shield is available. */

  if(TFT_ShieldDetect() == SHIELD_DETECTED)
  {
    /* Initialize the LCD */
    ADAFRUIT_802_LCD_Init(0, LCD_ORIENTATION_PORTRAIT);
    UTIL_LCD_SetFuncDriver(&LCD_Driver);

    /* Configure SD card */
    SDCard_Config();

    /* Display on TFT Images existing on SD card */
//    TFT_DisplayImages();

	/* Initialize the Joystick available on adafruit 1.8" TFT shield */
	ADAFRUIT_802_JOY_Init(JOY1, JOY_MODE_GPIO, JOY_ALL);


		//*****************************************************************************
		//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	#if ( FPU_ENABLED == 1 )
			uint8_t desc[3][18] = {
	#if ( FP_DOUBLE == 1 )
					{"lf_d6_1_45f       "},
					{"Double precision  "},
	#else
					{"lf_d6_1_45f       "},
					{"Single precision  "},
	#endif
					{"CM4 - FPU On      "}
			};
	#else
			uint8_t desc[3][18] = {
	#if ( FP_DOUBLE == 1 )
					{"lf_d6_1_45f       "},
					{"Double precision  "},
	#else
					{"lf_d6_1_45f       "},
					{"Single precision  "},
	#endif
					{"CM4 - FPU Off     "}
			};
	#endif

		//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
		//*****************************************************************************


		TFT_Display_Menu((uint8_t *)desc);

		/*Take HSEM */
		HAL_HSEM_FastTake(HSEM_ID_0);

		/*Release HSEM in order to notify the CPU2(CM4) ... run benchmark*/
		HAL_HSEM_Release(HSEM_ID_0,0);

		while (1) {
			if(HAL_HSEM_FastTake(HSEM_ID_1) == HAL_ERROR) {
				TFT_Display_Progress_1(n);
				if (n < 7) n++;
				else       n = 0;
				HAL_Delay(100);
			} else {
				HAL_HSEM_Release(HSEM_ID_1,0);
				break;
			}
		}

		// Performance in kFLOP/s

		int32_t kflops = *result;

		char kflops_str[18];

		if (kflops >= 0){
			itoa(kflops, kflops_str, 10);
			TFT_Display_result((uint8_t *) desc, (uint8_t *) kflops_str, (uint8_t *)"kFLOP/s PASSED");
		}
		else {
			itoa(-kflops, kflops_str, 10);
			TFT_Display_result((uint8_t *) desc, (uint8_t *) kflops_str, (uint8_t *)"kFLOP/s FAILED");
		}

		BSP_LED_Init(LED3);
		BSP_LED_On(LED3);
	}
	else /* Shield not mounted */
	{
		LED3_Blink();
	}


	/* Infinite loop */
	while (1)
	{
	}
}


/**
  * @brief  System Clock Configuration
  *         The system Clock is configured as follow :
  *            System Clock source            = PLL (HSE BYPASS)
  *            SYSCLK(Hz)                     = 400000000 (CPU Clock)
  *            HCLK(Hz)                       = 200000000 (Cortex-M4 CPU, Bus matrix Clocks)
  *            AHB Prescaler                  = 2
  *            D1 APB3 Prescaler              = 2 (APB3 Clock  100MHz)
  *            D2 APB1 Prescaler              = 2 (APB1 Clock  100MHz)
  *            D2 APB2 Prescaler              = 2 (APB2 Clock  100MHz)
  *            D3 APB4 Prescaler              = 2 (APB4 Clock  100MHz)
  *            HSE Frequency(Hz)              = 8000000
  *            PLL_M                          = 4
  *            PLL_N                          = 400
  *            PLL_P                          = 2
  *            PLL_Q                          = 4
  *            PLL_R                          = 2
  *            VDD(V)                         = 3.3
  *            Flash Latency(WS)              = 4
  * @param  None
  * @retval None
  */
static void SystemClock_Config(void)
{
  RCC_ClkInitTypeDef RCC_ClkInitStruct;
  RCC_OscInitTypeDef RCC_OscInitStruct;
  HAL_StatusTypeDef ret = HAL_OK;

  /*!< Supply configuration update enable */
  HAL_PWREx_ConfigSupply(PWR_DIRECT_SMPS_SUPPLY);

  /* The voltage scaling allows optimizing the power consumption when the device is
     clocked below the maximum system frequency, to update the voltage scaling value
     regarding system frequency refer to product datasheet.  */
  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);

  while(!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {}

  /* Enable HSE Oscillator and activate PLL with HSE as source */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
  RCC_OscInitStruct.HSEState = RCC_HSE_BYPASS;
  RCC_OscInitStruct.HSIState = RCC_HSI_OFF;
  RCC_OscInitStruct.CSIState = RCC_CSI_OFF;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
  RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;

  RCC_OscInitStruct.PLL.PLLM = 4;
  RCC_OscInitStruct.PLL.PLLN = 400;
  RCC_OscInitStruct.PLL.PLLFRACN = 0;
  RCC_OscInitStruct.PLL.PLLP = 2;
  RCC_OscInitStruct.PLL.PLLR = 2;
  RCC_OscInitStruct.PLL.PLLQ = 4;

  RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1VCOWIDE;
  RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1VCIRANGE_1;
  ret = HAL_RCC_OscConfig(&RCC_OscInitStruct);
  if(ret != HAL_OK)
  {
    Error_Handler();
  }

/* Select PLL as system clock source and configure  bus clocks dividers */
  RCC_ClkInitStruct.ClockType = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_D1PCLK1 | RCC_CLOCKTYPE_PCLK1 | \
                                 RCC_CLOCKTYPE_PCLK2  | RCC_CLOCKTYPE_D3PCLK1);

  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
  RCC_ClkInitStruct.SYSCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_HCLK_DIV2;
  RCC_ClkInitStruct.APB3CLKDivider = RCC_APB3_DIV2;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_APB1_DIV2;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_APB2_DIV2;
  RCC_ClkInitStruct.APB4CLKDivider = RCC_APB4_DIV2;
  ret = HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_4);
  if(ret != HAL_OK)
  {
    Error_Handler();
  }

  /*activate CSI clock mondatory for I/O Compensation Cell*/
  __HAL_RCC_CSI_ENABLE() ;

  /* Enable SYSCFG clock mondatory for I/O Compensation Cell */
  __HAL_RCC_SYSCFG_CLK_ENABLE() ;

  /* Enables the I/O Compensation Cell */
  HAL_EnableCompensationCell();

}

static void TFT_Display_Header(void)
{
	UTIL_LCD_Clear(UTIL_LCD_COLOR_WHITE);

	/* Set Menu font */
	UTIL_LCD_SetFont(&Font12);

	/* Set Text color */
	UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_RED);

	/* Display message */
	UTIL_LCD_DisplayStringAtLine(1, (uint8_t *)" NUCLEO-H755-Q    ");
	UTIL_LCD_DisplayStringAtLine(2, (uint8_t *)" Benchmark on CM4 ");
}

static void TFT_Display_Desc(uint8_t *desc)
{
	UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_BLUE);
	/* Display message */
	UTIL_LCD_DisplayStringAtLine(4, (uint8_t *)desc);
	UTIL_LCD_DisplayStringAtLine(5, (uint8_t *)(desc + 18));
	UTIL_LCD_DisplayStringAtLine(6, (uint8_t *)(desc + 36));
}


	static void TFT_Display_Menu(uint8_t *desc)
{
	  /* Set Menu font */
	  UTIL_LCD_SetFont(&Font12);

	  /* Set Back color */
	  UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_WHITE);

	TFT_Display_Header();
	TFT_Display_Desc(desc);

	/* Set Text color */
	UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_WHITE);
	/* Display message */
	UTIL_LCD_DisplayStringAtLine(11, (uint8_t *)"  Press JOY DOWN  ");
	UTIL_LCD_DisplayStringAtLine(12, (uint8_t *)"  to continue...  ");

	/* Wait for JOY_DOWN is pressed */

	//while (BSP_JOY_GetState() != JOY_DOWN)
	while (ADAFRUIT_802_JOY_GetState(JOY1) != JOY_DOWN)
	{}
	HAL_Delay(200);


	TFT_Display_Header();
	TFT_Display_Desc(desc);
	UTIL_LCD_DisplayStringAtLine(11, (uint8_t *)"Running ...       ");

}

static void TFT_Display_result(uint8_t *desc, uint8_t *result, uint8_t *unit)
{
	TFT_Display_Header();
	TFT_Display_Desc(desc);

	UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_BLACK);
	/* Display message */
	UTIL_LCD_DisplayStringAtLine(8,  (uint8_t *)"Result");
	UTIL_LCD_DisplayStringAtLine(9,  (uint8_t *)result);
	UTIL_LCD_DisplayStringAtLine(10, (uint8_t *)unit);
}


//static void TFT_Display_Progress(int current, uint32_t max)
//{
//	uint16_t pos;
//	float scale;
//
//	if (current == -1){
//		UTIL_LCD_DrawRect(2, 100, 122, 10, UTIL_LCD_COLOR_BLACK);
//	} else {
//		if ( max > 120 ) {
//
//			scale = 120.0f / (float)max;
//			current = (int)((float)(current) * scale);
//			pos = (uint16_t)(current + 1);
//
//		} else {
//			pos = (uint16_t)(((current + 1) * 120) / max);
//		}
//		UTIL_LCD_FillRect(3, 101, pos, 8, UTIL_LCD_COLOR_BLACK);
//	}
//}

static void TFT_Display_Progress_1(uint32_t n)
{

	switch (n) {
		case 0:
		case 4:
			UTIL_LCD_DisplayStringAtLine(9, (uint8_t *)"        |         ");
			break;
		case 1:
		case 5:
			UTIL_LCD_DisplayStringAtLine(9, (uint8_t *)"        /         ");
			break;
		case 2:
		case 6:
			UTIL_LCD_DisplayStringAtLine(9, (uint8_t *)"        -         ");
			break;
		case 3:
		case 7:
			UTIL_LCD_DisplayStringAtLine(9, (uint8_t *)"        \\        ");
			break;
	}

}


/**
  * @brief  SD Card Configuration.
  * @param  None
  * @retval None
  */
static void SDCard_Config(void)
{
  uint32_t counter = 0;

  if(FATFS_LinkDriver(&SD_Driver, SD_Path) == 0)
  {
    /* Initialize the SD mounted on adafruit 1.8" TFT shield */
    if(ADAFRUIT_802_SD_Init(0) != BSP_ERROR_NONE)
    {
      TFT_DisplayErrorMessage(BSP_SD_INIT_FAILED);
    }

    /* Check the mounted device */
    if(f_mount(&SD_FatFs, (TCHAR const*)"/", 0) != FR_OK)
    {
      TFT_DisplayErrorMessage(FATFS_NOT_MOUNTED);
    }
    else
    {
      /* Initialize the Directory Files pointers (heap) */
      for (counter = 0; counter < MAX_BMP_FILES; counter++)
      {
        pDirectoryFiles[counter] = malloc(11);
      }
    }
  }
}

/**
  * @brief  Displays adequate message on TFT available on adafruit 1.8" TFT shield
  * @param  message: Error message to be displayed on the LCD.
  *   This parameter can be one of following values:
  *     @arg SD_CARD_NOT_FORMATTED: SD CARD is not FAT formatted
  *     @arg SD_CARD_FILE_NOT_SUPPORTED: File is not supported
  *     @arg SD_CARD_OPEN_FAIL: Failure to open directory
  *     @arg FATFS_NOT_MOUNTED: FatFs is not mounted
  * @retval None
  */
static void TFT_DisplayErrorMessage(uint8_t message)
{
  /* LCD Clear */
  UTIL_LCD_Clear(UTIL_LCD_COLOR_WHITE);
  /* Set Error Message Font */
  UTIL_LCD_SetFont(&Font12);
  /* Set Text and Back colors */
  UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_ST_GRAY);
  UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_RED);

  if(message == SD_CARD_NOT_FORMATTED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAt(0,50, (uint8_t*)" SD Card is not    ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,60, (uint8_t*)" FAT formatted.    ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,70, (uint8_t*)" Please Format the ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,80, (uint8_t*)" microSD card.     ", LEFT_MODE);
    while (1)
    {
    }
  }
  if(message == SD_CARD_FILE_NOT_SUPPORTED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAt(0,50, (uint8_t*)"                   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,60, (uint8_t*)" File type is not  ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,70, (uint8_t*)" supported.        ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,80, (uint8_t*)"                   ", LEFT_MODE);
    while(1)
    {
    }
  }
  if(message == SD_CARD_OPEN_FAIL)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAt(0,50, (uint8_t*)"                   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,60, (uint8_t*)" Open directory    ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,70, (uint8_t*)" fails.            ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,80, (uint8_t*)"                   ", LEFT_MODE);
    while(1)
    {
    }
  }
  if(message == FATFS_NOT_MOUNTED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAt(0,50, (uint8_t*)"                   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,60, (uint8_t*)" Cannot mount      ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,70, (uint8_t*)" FatFs on Drive.   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,80, (uint8_t*)"                   ", LEFT_MODE);
    while (1)
    {
    }
  }
  if(message == BSP_SD_INIT_FAILED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAt(0,50, (uint8_t*)"                   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,60, (uint8_t*)" SD Init           ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,70, (uint8_t*)" fails.            ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,80, (uint8_t*)"                   ", LEFT_MODE);
    while(1)
    {
    }
  }
  if(message == SD_CARD_NO_FILE)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAt(0,50, (uint8_t*)"                   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,60, (uint8_t*)" NO image found     ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,70, (uint8_t*)" in the SD card.   ", LEFT_MODE);
    UTIL_LCD_DisplayStringAt(0,80, (uint8_t*)"                   ", LEFT_MODE);
    while(1)
    {
    }
  }
}

///**
//  * @brief  Blinks LED1 with two frequencies depending on User press button.
//  * @param  None
//  * @retval None
//  */
//static void LED1_Blink(void)
//{
//  /* Configure LED1 on Nucleo */
//  BSP_LED_Init(LED1);
//
//  /* Configure the User Button in EXTI Mode */
//  BSP_PB_Init(BUTTON_USER, BUTTON_MODE_EXTI);
//
//  /* Initiate BlinkSpeed variable */
//  BlinkSpeed = 0;
//
//  /*Take HSEM */
//  HAL_HSEM_FastTake(HSEM_ID_0);
//
//  /* Infinite loop */
//  while(1)
//  {
//    /*Release HSEM in order to notify the CPU2(CM4)*/
//    HAL_HSEM_Release(HSEM_ID_0,0);
//
//    /* Test on blink speed */
//    if(BlinkSpeed == 0)
//    {
//      BSP_LED_Toggle(LED1);
//      /* Wait for 500ms */
//      HAL_Delay(500);
//    }
//    else if(BlinkSpeed == 1)
//    {
//      BSP_LED_Toggle(LED1);
//      /* Wait for 100ms */
//      HAL_Delay(100);
//    }
//    else if(BlinkSpeed == 2)
//    {
//      BSP_LED_Toggle(LED1);
//      /* Wait for 50ms */
//      HAL_Delay(50);
//    }
//    /*Take HSEM */
//    HAL_HSEM_FastTake(HSEM_ID_0);
//  }
//}

/**
  * @brief  Blinks LED3 with two frequencies depending on User press button.
  * @param  None
  * @retval None
  */
static void LED3_Blink(void)
{
  /* Configure LED3 on Nucleo */
  BSP_LED_Init(LED3);

  /* Configure the User Button in EXTI Mode */
  BSP_PB_Init(BUTTON_USER, BUTTON_MODE_EXTI);

  /* Initiate BlinkSpeed variable */
  BlinkSpeed = 0;

  /*Take HSEM */
  HAL_HSEM_FastTake(HSEM_ID_0);

  /* Infinite loop */
  while(1)
  {
    /*Release HSEM in order to notify the CPU2(CM4)*/
    HAL_HSEM_Release(HSEM_ID_0,0);

    /* Test on blink speed */
    if(BlinkSpeed == 0)
    {
      BSP_LED_Toggle(LED3);
      /* Wait for 500ms */
      HAL_Delay(500);
    }
    else if(BlinkSpeed == 1)
    {
      BSP_LED_Toggle(LED3);
      /* Wait for 100ms */
      HAL_Delay(100);
    }
    else if(BlinkSpeed == 2)
    {
      BSP_LED_Toggle(LED3);
      /* Wait for 50ms */
      HAL_Delay(50);
    }
    /*Take HSEM */
    HAL_HSEM_FastTake(HSEM_ID_0);
  }
}

///**
//  * @brief  Switch LED1 on - permanent light.
//  * @param  None
//  * @retval None
//  */
//static void LED1_On(void)
//{
//	BSP_LED_Init(LED1);
//	BSP_LED_On(LED1);
//}


/**
  * @brief  EXTI line detection callbacks.
  * @param  GPIO_Pin: Specifies the pins connected EXTI line
  * @retval None
  */
void BSP_PB_Callback(Button_TypeDef Button)
{
  if(BlinkSpeed == 2)
  {
    BlinkSpeed = 0;
  }
  else
  {
    BlinkSpeed ++;
  }
}


/**
  * @brief  Check the availability of adafruit 1.8" TFT shield on top of STM32NUCLEO
  *         board. This is done by reading the state of IO PF.03 pin (mapped to
  *         JoyStick available on adafruit 1.8" TFT shield). If the state of PF.03
  *         is high then the adafruit 1.8" TFT shield is available.
  * @param  None
  * @retval SHIELD_DETECTED: 1.8" TFT shield is available
  *         SHIELD_NOT_DETECTED: 1.8" TFT shield is not available
  */
static ShieldStatus TFT_ShieldDetect(void)
{
  GPIO_InitTypeDef  GPIO_InitStruct;

  /* Enable GPIO clock */
  ADAFRUIT_802_ADCx_GPIO_CLK_ENABLE();
  GPIO_InitStruct.Pin = ADAFRUIT_802_ADCx_GPIO_PIN ;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_PULLDOWN;
  HAL_GPIO_Init(ADAFRUIT_802_ADCx_GPIO_PORT , &GPIO_InitStruct);

  if(HAL_GPIO_ReadPin(ADAFRUIT_802_ADCx_GPIO_PORT, ADAFRUIT_802_ADCx_GPIO_PIN) != 0)
  {
    return SHIELD_DETECTED;
  }
  else
  {
    return SHIELD_NOT_DETECTED;
  }
}



/**
* @brief  CPU L1-Cache enable.
* @param  None
* @retval None
*/
static void CPU_CACHE_Enable(void)
{
  /* Enable I-Cache */
  SCB_EnableICache();

  /* Enable D-Cache */
  SCB_EnableDCache();
}

/**
  * @brief  This function is executed in case of error occurrence.
  * @param  None
  * @retval None
  */
static void Error_Handler(void)
{
  while(1)
  {
  }
}


/**
  * @brief  Configure the MPU attributes
  * @param  None
  * @retval None
  */
static void MPU_Config(void)
{
  MPU_Region_InitTypeDef MPU_InitStruct;

  /* Disable the MPU */
  HAL_MPU_Disable();

  /* Configure the MPU as Strongly ordered for not defined regions */
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.BaseAddress = 0x00;
  MPU_InitStruct.Size = MPU_REGION_SIZE_4GB;
  MPU_InitStruct.AccessPermission = MPU_REGION_NO_ACCESS;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_NOT_CACHEABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.SubRegionDisable = 0x87;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;

  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  /* Enable the MPU */
  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

