	/**
  ******************************************************************************
  * @file    Demonstrations/CM4/Src/main.c
  * @author  MCD Application Team
  * @brief   main program for Cortex-M4 
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2018 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup STM32H7xx_HAL_Demonstrations
  * @{
  */

/** @addtogroup Demo
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
typedef enum
{
  SHIELD_NOT_DETECTED = 0,
  SHIELD_DETECTED
}ShieldStatus;

/* Private define ------------------------------------------------------------*/
#define HSEM_ID_0 (0U) /* HW semaphore 0*/
#define HSEM_ID_1 (1U) /* HW semaphore 1*/
#define HSEM_ID_2 (2U) /* HW semaphore 2*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
volatile uint32_t Notif_Recieved = 0;
/* Private function prototypes -----------------------------------------------*/

/* Private functions ---------------------------------------------------------*/
//static ShieldStatus TFT_ShieldDetect(void);
static int32_t Run_Benchmark(void);

/* benchmark parameters */

#define SH_RESULT_ADDR 		0x30000000
volatile int32_t *result = SH_RESULT_ADDR;

//*****************************************************************************
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

/* benchmark function */

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
//*****************************************************************************

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
 /*HW semaphore Clock enable*/
  __HAL_RCC_HSEM_CLK_ENABLE();
 
  /* Enable HSEM Interrupt */
  HAL_NVIC_SetPriority(HSEM2_IRQn, 10, 0);
  HAL_NVIC_EnableIRQ(HSEM2_IRQn);

  /* Activate HSEM notification for Cortex-M4*/
  HAL_HSEM_ActivateNotification(__HAL_HSEM_SEMID_TO_MASK(HSEM_ID_0));
  
  /* 
    Domain D2 goes to STOP mode (Cortex-M4 in deep-sleep) waiting for Cortex-M7 to
    perform system initialization (system clock config, external memory configuration.. )   
  */
  HAL_PWREx_ClearPendingEvent();
  HAL_PWREx_EnterSTOPMode(PWR_MAINREGULATOR_ON, PWR_STOPENTRY_WFE, PWR_D2_DOMAIN);

  /* Clear HSEM flag */
  __HAL_HSEM_CLEAR_FLAG(__HAL_HSEM_SEMID_TO_MASK(HSEM_ID_0));

 /* STM32H7xx HAL library initialization:
       - Systick timer is configured by default as source of time base, but user 
         can eventually implement his proper time base source (a general purpose 
         timer for example or other time source), keeping in mind that Time base 
         duration should be kept 1ms since PPP_TIMEOUT_VALUEs are defined and 
         handled in milliseconds basis.
       - Set NVIC Group Priority to 4
       - Low Level Initialization
     */
  HAL_Init();

  /* Add Cortex-M4 user application code here */

	/* Initialize LED 2 */
	BSP_LED_Init(LED2);

	/* Wait for Notification from CPU1(CM7)*/
	while( Notif_Recieved == 0) {}

	/* Notification received from CPU1(CM7)*/

	Notif_Recieved = 0;
	/* HSEM Activate Notification*/
	HAL_HSEM_ActivateNotification(__HAL_HSEM_SEMID_TO_MASK(HSEM_ID_0));

	/*Take HSEM */
	HAL_HSEM_FastTake(HSEM_ID_1);

	BSP_LED_On(LED2);

	/* Wait for Notification from CPU1(CM7)*/
	while( Notif_Recieved == 0) {}
	Notif_Recieved = 0;

	/* HSEM Activate Notification*/
	HAL_HSEM_ActivateNotification(__HAL_HSEM_SEMID_TO_MASK(HSEM_ID_0));

	BSP_LED_Off(LED2);


	//*****************************************************************************
	//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	*result = Run_Benchmark();

	//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
	//*****************************************************************************


	BSP_LED_On(LED2);

	/*Release HSEM in order to notify the CPU2(CM4) ... run benchmark*/
	HAL_HSEM_Release(HSEM_ID_1,0);
	HAL_HSEM_ActivateNotification(__HAL_HSEM_SEMID_TO_MASK(HSEM_ID_1));

	while(1){}

}
  
//*****************************************************************************
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

// System description and I/O data generated in Scilab

static int32_t Run_Benchmark(void)
{

    int i, j, k, compare;

	// System description and I/O data generated in Scilab
	#include "sci_albm.h"
	#include "sci_albdm.h"
	#include "sci_alfm.h"
	#include "sci_betm.h"
	#include "sci_betzm.h"
	#include "sci_bkbm.h"
	#include "sci_bkfm.h"
	#include "sci_bkzm.h"
	#include "sci_dfm.h"
	#include "sci_ebdpm.h"
	#include "sci_ebpm.h"
	#include "sci_edm.h"
	#include "sci_efm.h"
	#include "sci_efpm.h"
	#include "sci_em.h"
	#include "sci_epm.h"
	#include "sci_fiexm.h"
	#include "sci_fim.h"
	#include "sci_gpm.h"
	#include "sci_gdm.h"
	#include "sci_lam.h"
	#include "sci_md45_d.h"
	#include "sci_md45_i.h"
	#include "sci_md45_t.h"
	#include "sci_N.h"
	#include "sci_NF.h"
	#include "sci_NT.h"
	#include "sci_rm.h"
	#include "sci_u.h"
	#include "sci_udm.h"
	#include "sci_vdm.h"
	#include "sci_vm.h"
	#include "sci_y.h"
	#include "sci_ydm.h"
	#include "sci_zem.h"
	#include "sci_zum.h"
	#include "sci_zvm.h"
	#include "sci_zym.h"

	 extern void 	md45f_lat(	float * sci_u, 			float * sci_y, 		float * sci_vm, 		float * sci_em, 		float * sci_efm,
								float * sci_udm,		float * sci_ydm,	float * sci_vdm,		float * sci_edm,		float * sci_zum,
								float * sci_zym,		float * sci_zvm,	float * sci_zem,		float * sci_rm,			float * sci_dfm,
								float * sci_fim,		float * sci_lam,	float * sci_betm,		float * sci_bkfm,		float * sci_bkbm,
								float * sci_betzm,		float * sci_bkzm,	float * sci_efpm,		float * sci_ebpm,		float * sci_ebdpm,
								float * sci_epm,		float * sci_alfm,	float * sci_albm,		float * sci_albdm, 		float * sci_gpm,
								float * sci_gdm,		float * sci_fiexm,	float * sci_md45_i,		float * sci_md45_d,		float * sci_md45_t,
								float * sci_N, 			float * sci_NF, 	float * sci_NT);

	uint32_t score_fpu = 0;
	uint32_t tickstart = 0;
	int32_t kflops = 0;

	// init progress bar
	//	TFT_Display_Progress(-1, (uint32_t) sci_N[0]);

	// time
	tickstart = HAL_GetTick();

	// System model: Input: sci_u; Output: sci_y
	// System identification: Recursive identification steps performed: sci_N; Identification System Order: sci_md51_i[11]

	md45f_lat(	(float *) sci_u, 		(float *) sci_y, 	(float *) sci_vm, 		(float *) sci_em, 		(float *) sci_efm,
				(float *) sci_udm,		(float *) sci_ydm,	(float *) sci_vdm,		(float *) sci_edm,		(float *) sci_zum,
				(float *) sci_zym,		(float *) sci_zvm,	(float *) sci_zem,		(float *) sci_rm,		(float *) sci_dfm,
				(float *) sci_fim,		(float *) sci_lam,	(float *) sci_betm,		(float *) sci_bkfm,		(float *) sci_bkbm,
				(float *) sci_betzm,	(float *) sci_bkzm,	(float *) sci_efpm,		(float *) sci_ebpm,		(float *) sci_ebdpm,
				(float *) sci_epm,		(float *) sci_alfm,	(float *) sci_albm,		(float *) sci_albdm, 	(float *) sci_gpm,
				(float *) sci_gdm,		(float *) sci_fiexm,	(float *) sci_md45_i,	(float *) sci_md45_d,	(float *) sci_md45_t,
				(float *) sci_N, 		(float *) sci_NF, 	(float *) sci_NT);

	// measure time in ms
	score_fpu += (uint32_t)(HAL_GetTick() - tickstart);

	//TFT_Display_Progress((uint32_t)  sci_N[0], (uint32_t)  sci_N[0]);

	//copmpare ref with test
	//    ref  test:
	//efm   9    13
	//em    8    12
	//pm2   7    11
    //pm1   5    10

	compare = 0;

	// test for efm filtration error
	for (i=0; i<(uint32_t)  sci_N[0]; i++){
		j =  9 * ((uint32_t)   sci_N[0]) + i;
		k = 13 * ((uint32_t)   sci_N[0]) + i;
		if (sci_md45_t[j] > - 0.1 &&sci_md45_t[j] < 0.1){
			if (sci_md45_t[j]-sci_md45_t[k] < -0.001 || sci_md45_t[j]-sci_md45_t[k]> 0.001){
				compare = 1;
			}
		}
		else{
			if (sci_md45_t[k]/sci_md45_t[j] < 0.999 || sci_md45_t[k]-sci_md45_t[j]> 1.001){
				compare = 1;
			}
		}
	}

	// test for em prediction error
	for (i=0; i<(uint32_t)  sci_N[0]; i++){
		j =  8 * ((uint32_t)  sci_N[0]) + i;
		k = 12 * ((uint32_t)  sci_N[0]) + i;
		if ( sci_md45_t[j] > - 0.1 && sci_md45_t[j] < 0.1){
			if (sci_md45_t[j]-sci_md45_t[k] < -0.001 || sci_md45_t[j]-sci_md45_t[k]> 0.001){
				compare = 1;
			}
		}
		else{
			if (sci_md45_t[k]/sci_md45_t[j] < 0.999 || sci_md45_t[k]-sci_md45_t[j]> 1.001){
				compare = 1;
			}
		}
	}

	// compute and return kFLOP/s
	kflops = (int32_t)( ((   32 * ((uint32_t) sci_md45_i[11] + 1)
			               + 14
						  )  * (uint32_t)  sci_N[0]) / ((uint32_t) score_fpu) );

	if (compare == 1){
		kflops = - kflops;
	}

	return kflops;
	//return score_fpu;
	//positive PASSED, negative FAILED
}

  //<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
  //*****************************************************************************



/**
  * @brief Semaphore Released Callback.
  * @param SemMask: Mask of Released semaphores
  * @retval None
  */
void HAL_HSEM_FreeCallback(uint32_t SemMask)
{
  Notif_Recieved = 1;
} 

/**
  * @brief  Check the availability of adafruit 1.8" TFT shield on top of STM32NUCLEO
  *         board. This is done by reading the state of IO PF.03 pin (mapped to
  *         JoyStick available on adafruit 1.8" TFT shield). If the state of PF.03
  *         is high then the adafruit 1.8" TFT shield is available.
  * @param  None
  * @retval SHIELD_DETECTED: 1.8" TFT shield is available
  *         SHIELD_NOT_DETECTED: 1.8" TFT shield is not available
  */
//static ShieldStatus TFT_ShieldDetect(void)
//{
//  GPIO_InitTypeDef  GPIO_InitStruct;
//
//  /* Enable GPIO clock */
//  ADAFRUIT_802_ADCx_GPIO_CLK_ENABLE();
//
//  GPIO_InitStruct.Pin = ADAFRUIT_802_ADCx_GPIO_PIN ;
//  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
//  GPIO_InitStruct.Pull = GPIO_PULLDOWN;
//  HAL_GPIO_Init(ADAFRUIT_802_ADCx_GPIO_PORT , &GPIO_InitStruct);
//
//  if(HAL_GPIO_ReadPin(ADAFRUIT_802_ADCx_GPIO_PORT, ADAFRUIT_802_ADCx_GPIO_PIN) != 0)
//  {
//    return SHIELD_DETECTED;
//  }
//  else
//  {
//    return SHIELD_NOT_DETECTED;
//  }
//}

#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/**
  * @}
  */

/**
  * @}
  */

