/**
  ******************************************************************************
  * @file    adafruit_802_v2.c
  * @author  MCD Application Team
  * @brief   This file provides set of firmware functions to manage:
  *          - Joystick available on Adafruit 1.8" TFT LCD shield (reference ID 802_V2)
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2018 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "adafruit_802.h"
#include "adafruit_802_v2.h"

/** @addtogroup BSP
  * @{
  */

/** @addtogroup ADAFRUIT_802_V2
  * @{
  */

/** @addtogroup ADAFRUIT_802_V2_LOW_LEVEL
  * @{
  */

/** @addtogroup ADAFRUIT_802_LOW_LEVEL_Exported_Functions
  * @{
  */


/**
  * @brief  Initialize SeeSaw chip on adafruit 1.8" TFT shield V2
  * @param  I2C core handler, the core has to be already initialized
  * @retval 0 on pass
  *         negative number on failure
  */
int32_t ADAFRUIT_802_V2_Init(I2C_HandleTypeDef *I2cHandle)
{
    uint8_t i2c_d [6];

    // reset all seesaw regs to defaults
    i2c_d[0] = 0x00;
    i2c_d[1] = 0x7F;
    i2c_d[2] = 0xFF;
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 3, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -1;
        }
    }
    HAL_Delay(500);

#if 0
    // read ID
    i2c_d[0] = 0x00;
    i2c_d[1] = 0x01;
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 2, 1000) != HAL_OK) {
    if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
      Error_Handler();
    }
    }
    while(HAL_I2C_Master_Receive(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 1, 1000) != HAL_OK) {
      if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
        Error_Handler();
      }
  }
#endif


    // enable backlight
    i2c_d[0] = 0x08;
    i2c_d[1] = 0x01;
    i2c_d[2] = 0x00;
    i2c_d[3] = 0xFF;
    i2c_d[4] = 0xFF;
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 5, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -2;
        }
    }


    // set reset pin as output
    i2c_d[0] = 0x01;
    i2c_d[1] = 0x02;
    i2c_d[2] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN >> 24);
    i2c_d[3] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN >> 16);
    i2c_d[4] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN >> 8);
    i2c_d[5] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN);
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 6, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -3;
        }
    }


    // LCD reset
    i2c_d[0] = 0x01;
    i2c_d[1] = 0x05;
    i2c_d[2] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN >> 24);
    i2c_d[3] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN >> 16);
    i2c_d[4] = (uint8_t)(ADAFRUIT_802_V2_RESET_PIN >> 8);
    i2c_d[5] = (uint8_t)ADAFRUIT_802_V2_RESET_PIN;
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 6, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -4;
        }
    }

    HAL_Delay(20);

    // set seesaw pins as inputs - buttons and JOY
    i2c_d[0] = 0x01;
    i2c_d[1] = 0x03;
    i2c_d[2] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 24);
    i2c_d[3] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 16);
    i2c_d[4] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 8);
    i2c_d[5] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL);
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 6, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -5;
        }
    }

    // set pin val to high, buttons are active in low level
    i2c_d[0] = 0x01;
    i2c_d[1] = 0x05;
    i2c_d[2] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 24);
    i2c_d[3] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 16);
    i2c_d[4] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 8);
    i2c_d[5] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL);
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 6, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -6;
        }
    }

    // enable internal PULLUPs
    i2c_d[0] = 0x01;
    i2c_d[1] = 0x0B;
    i2c_d[2] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 24);
    i2c_d[3] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 16);
    i2c_d[4] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL >> 8);
    i2c_d[5] = (uint8_t)(ADAFRUIT_802_V2_BUTTON_ALL);
    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 6, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -7;
        }
    }

    return 0;
}


/**
  * @brief  Returns the Joystick key pressed of adafruit 1.8" TFT shield V2.
  * @param  I2C core handler, the core has to be already initialized
  * @retval JOYState_TypeDef: Code of the Joystick key pressed.
  *         negative number on failure
  */
int32_t ADAFRUIT_802_V2_JOY_GetState(I2C_HandleTypeDef *I2cHandle)
{
    int32_t state;
    uint8_t i2c_d[4];
    uint32_t btn_state;

    i2c_d[0] = 0x01;
    i2c_d[1] = 0x04;

    while(HAL_I2C_Master_Transmit(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 2, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
            //Error_Handler();
            return -1;
        }
    }
    while(HAL_I2C_Master_Receive(I2cHandle, (uint16_t)ADAFRUIT_802_V2_I2C_ADDR, i2c_d, 4, 1000) != HAL_OK) {
        if (HAL_I2C_GetError(I2cHandle) != HAL_I2C_ERROR_AF) {
//             Error_Handler();
            return -2;
        }
    }

    btn_state = ~( (uint32_t)i2c_d[0] << 24 | (uint32_t)i2c_d[1] << 16 | (uint32_t)i2c_d[2] << 8 | (uint32_t)i2c_d[3] );

    if( btn_state & ADAFRUIT_802_V2_BUTTON_DOWN ) {
        state = (uint32_t)JOY_DOWN;
    } else if ( btn_state & ADAFRUIT_802_V2_BUTTON_RIGHT ) {
        state = (uint32_t)JOY_RIGHT;
    } else if ( btn_state & ADAFRUIT_802_V2_BUTTON_IN ) {
        state = (uint32_t)JOY_SEL;
    } else if ( btn_state & ADAFRUIT_802_V2_BUTTON_UP ) {
        state = (uint32_t)JOY_UP;
    } else if ( btn_state & ADAFRUIT_802_V2_BUTTON_LEFT ) {
        state = (uint32_t)JOY_LEFT;
    } else {
        state = (uint32_t)JOY_NONE;
    }

    return state;
}



/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
