/**
  ******************************************************************************
  * @file    Demonstrations/Src/main.c
  * @author  MCD Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * Copyright (c) 2019 STMicroelectronics.
  * All rights reserved.
  *
  * This software is licensed under terms that can be found in the LICENSE file
  * in the root directory of this software component.
  * If no LICENSE file comes with this software, it is provided AS-IS.
  *
  ******************************************************************************
  */


/* Includes ------------------------------------------------------------------*/
#include "main.h"

/** @addtogroup STM32H7xx_HAL_Examples
  * @{
  */

/** @addtogroup Demonstrations
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
typedef enum
{
  SHIELD_NOT_DETECTED = 0,
  SHIELD_DETECTED
}ShieldStatus;

/* Private define ------------------------------------------------------------*/
#define SD_CARD_NOT_FORMATTED                    0
#define SD_CARD_FILE_NOT_SUPPORTED               1
#define SD_CARD_OPEN_FAIL                        2
#define FATFS_NOT_MOUNTED                        3
#define BSP_SD_INIT_FAILED                       4
#define SD_CARD_NO_FILE                          5

#define POSITION_X_BITMAP                        0
#define POSITION_Y_BITMAP                        0

#define FPU_ENABLED								 1
#define FP_DOUBLE								 0

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint8_t BlinkSpeed = 0, str[20];
__IO uint8_t JoystickValue = 0;
char* pDirectoryFiles[MAX_BMP_FILES];
FATFS SD_FatFs;  /* File system object for SD card logical drive */
char SD_Path[4]; /* SD card logical drive path */

/* Private function prototypes -----------------------------------------------*/
static void MPU_Config(void);
static void SystemClock_Config(void);
static void CPU_CACHE_Enable(void);
static void Error_Handler(void);
static void LED2_Blink(void);
static ShieldStatus TFT_ShieldDetect(void);
static void SDCard_Config(void);
static void TFT_DisplayErrorMessage(uint8_t message);
//static void TFT_DisplayMenu(void);
//static void TFT_DisplayImages(void);

static void TFT_Display_Header(void);
static void TFT_Display_Desc(uint8_t *desc);
static void TFT_Display_Menu(uint8_t *desc);
static void TFT_Display_result(uint8_t *desc, uint8_t *result, uint8_t *unit);
static void TFT_Display_Progress(int current, uint32_t max);

static int32_t Run_Benchmark(void);

//*****************************************************************************
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

/* benchmark parameters */

//<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
//*****************************************************************************



/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
	  /* Configure the MPU attributes */
	  MPU_Config();

  /* Enable the CPU Cache */
  CPU_CACHE_Enable();

  /* STM32H7xx HAL library initialization:
       - Configure the Systick to generate an interrupt each 1 msec
       - Set NVIC Group Priority to 4
       - Low Level Initialization
     */
  HAL_Init();

  /* Configure the system clock to 280 MHz */
  SystemClock_Config();

  /* Check the availability of adafruit 1.8" TFT shield on top of STM32NUCLEO
     board. This is done by reading the state of IO PF.03 pin (mapped to JoyStick
     available on adafruit 1.8" TFT shield). If the state of PF.03 is high then
     the adafruit 1.8" TFT shield is available. */
  if(TFT_ShieldDetect() == SHIELD_DETECTED)
  {
    /* Initialize the LCD */
    ADAFRUIT_802_LCD_Init(0, LCD_ORIENTATION_PORTRAIT);
    UTIL_LCD_SetFuncDriver(&LCD_Driver);

    /* Configure SD card */
    SDCard_Config();

    /* Display on TFT Images existing on SD card */
//    TFT_DisplayImages();

	/* Initialize the Joystick available on adafruit 1.8" TFT shield */
	ADAFRUIT_802_JOY_Init(JOY1, JOY_MODE_GPIO, JOY_ALL);

	//*****************************************************************************
	//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	#if ( FPU_ENABLED == 1 )
		uint8_t desc[3][18] = {
				{"lf_d3_1_45f       "},
	#if ( FP_DOUBLE == 1 )
				{"Double precision  "},
	#else
				{"Single precision  "},
	#endif
				{"CM7 - FPU On      "}
		};
	#else
		uint8_t desc[3][18] = {
				{"lf_d3_1_45f       "},
	#if ( FP_DOUBLE == 1 )
				{"Double precision  "},
	#else
				{"Single precision  "},
	#endif
				{"CM7 - FPU Off     "}
		};
	#endif


    TFT_Display_Menu((uint8_t *)desc);

	int32_t kflops = Run_Benchmark();

	// Performance in MFLOP/s
	char kflops_str[18];
	if (kflops >= 0){
		itoa(kflops, kflops_str, 10);
		TFT_Display_result((uint8_t *) desc, (uint8_t *) kflops_str, (uint8_t *)"kFLOP/s PASSED");
	}
	else {
		itoa(-kflops, kflops_str, 10);
		TFT_Display_result((uint8_t *) desc, (uint8_t *) kflops_str, (uint8_t *)"kFLOP/s FAILED");
	}

	BSP_LED_Init(LED2);
	BSP_LED_On(LED2);

  }
  else /* Shield not mounted */
  {
    LED2_Blink();
  }

  /* Infinite loop */
  while (1)
  {
  }
}


//*****************************************************************************
//>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


static int32_t Run_Benchmark(void)
{
	int i, j, k, compare;

	// System description and I/O data generated in Scilab
#include "sci_albm.h"
#include "sci_albdm.h"
#include "sci_alfm.h"
#include "sci_betm.h"
#include "sci_betzm.h"
#include "sci_bkbm.h"
#include "sci_bkfm.h"
#include "sci_bkzm.h"
#include "sci_dfm.h"
#include "sci_ebdpm.h"
#include "sci_ebpm.h"
#include "sci_edm.h"
#include "sci_efm.h"
#include "sci_efpm.h"
#include "sci_em.h"
#include "sci_epm.h"
#include "sci_fiexm.h"
#include "sci_fim.h"
#include "sci_gpm.h"
#include "sci_gdm.h"
#include "sci_lam.h"
#include "sci_md45_d.h"
#include "sci_md45_i.h"
#include "sci_md45_t.h"
#include "sci_N.h"
#include "sci_NF.h"
#include "sci_NT.h"
#include "sci_rm.h"
#include "sci_u.h"
#include "sci_udm.h"
#include "sci_vdm.h"
#include "sci_vm.h"
#include "sci_y.h"
#include "sci_ydm.h"
#include "sci_zem.h"
#include "sci_zum.h"
#include "sci_zvm.h"
#include "sci_zym.h"

 extern void 	md45f_lat(	float * sci_u, 			float * sci_y, 		float * sci_vm, 		float * sci_em, 		float * sci_efm,
							float * sci_udm,		float * sci_ydm,	float * sci_vdm,		float * sci_edm,		float * sci_zum,
							float * sci_zym,		float * sci_zvm,	float * sci_zem,		float * sci_rm,			float * sci_dfm,
							float * sci_fim,		float * sci_lam,	float * sci_betm,		float * sci_bkfm,		float * sci_bkbm,
							float * sci_betzm,		float * sci_bkzm,	float * sci_efpm,		float * sci_ebpm,		float * sci_ebdpm,
							float * sci_epm,		float * sci_alfm,	float * sci_albm,		float * sci_albdm, 		float * sci_gpm,
							float * sci_gdm,		float * sci_fiexm,	float * sci_md45_i,		float * sci_md45_d,		float * sci_md45_t,
							float * sci_N, 			float * sci_NF, 	float * sci_NT);

  	uint32_t score_fpu = 0;
	uint32_t tickstart = 0;
	int32_t kflops = 0;

 	// init progress bar
	TFT_Display_Progress(-1, (uint32_t) sci_N[0]);
	// time
	tickstart = HAL_GetTick();

	// System model: Input: sci_u; Output: sci_y
	// System identification: Recursive identification steps performed: sci_N;

	md45f_lat(	(float *) sci_u, 		(float *) sci_y, 	(float *) sci_vm, 		(float *) sci_em, 		(float *) sci_efm,
				(float *) sci_udm,		(float *) sci_ydm,	(float *) sci_vdm,		(float *) sci_edm,		(float *) sci_zum,
				(float *) sci_zym,		(float *) sci_zvm,	(float *) sci_zem,		(float *) sci_rm,		(float *) sci_dfm,
				(float *) sci_fim,		(float *) sci_lam,	(float *) sci_betm,		(float *) sci_bkfm,		(float *) sci_bkbm,
				(float *) sci_betzm,	(float *) sci_bkzm,	(float *) sci_efpm,		(float *) sci_ebpm,		(float *) sci_ebdpm,
				(float *) sci_epm,		(float *) sci_alfm,	(float *) sci_albm,		(float *) sci_albdm, 	(float *) sci_gpm,
				(float *) sci_gdm,		(float *) sci_fiexm,	(float *) sci_md45_i,	(float *) sci_md45_d,	(float *) sci_md45_t,
				(float *) sci_N, 		(float *) sci_NF, 	(float *) sci_NT);

	// measure time in ms
	score_fpu += (uint32_t)(HAL_GetTick() - tickstart);

	TFT_Display_Progress((uint32_t)  sci_N[0], (uint32_t)  sci_N[0]);

	//copmpare ref with test
	//    ref  test:
	//efm   9    13
	//em    8    12
	//pm2   7    11
    //pm1   5    10

	compare = 0;

	// test for efm filtration error
	for (i=0; i<(uint32_t)  sci_N[0]; i++){
		j =  9 * ((uint32_t)  sci_N[0]) + i;
		k = 13 * ((uint32_t)  sci_N[0]) + i;
		if ( sci_md45_t[j] > - 0.1 && sci_md45_t[j] < 0.1){
			if (sci_md45_t[j]-sci_md45_t[k] < -0.001 || sci_md45_t[j]-sci_md45_t[k]> 0.001){
				compare = 1;
			}
		}
		else{
			if (sci_md45_t[k]/sci_md45_t[j] < 0.999 || sci_md45_t[k]-sci_md45_t[j]> 1.001){
				compare = 1;
			}
		}
	}

	// test for em prediction error
	for (i=0; i<(uint32_t)  sci_N[0]; i++){
		j =  8 * ((uint32_t)  sci_N[0]) + i;
		k = 12 * ((uint32_t)  sci_N[0]) + i;
		if ( sci_md45_t[j] > - 0.1 && sci_md45_t[j] < 0.1){
			if (sci_md45_t[j]-sci_md45_t[k] < -0.001 || sci_md45_t[j]-sci_md45_t[k]> 0.001){
				compare = 1;
			}
		}
		else{
			if (sci_md45_t[k]/sci_md45_t[j] < 0.999 || sci_md45_t[k]-sci_md45_t[j]> 1.001){
				compare = 1;
			}
		}
	}

	// compute and return kFLOP/s
	kflops = (int32_t)( ((   32 * ((uint32_t) sci_md45_i[11] + 1)
			               + 14
						  )  * (uint32_t)  sci_N[0]) / ((uint32_t) score_fpu) );

	if (compare == 1){
		kflops = - kflops;
	}

	return kflops;
	//return score_fpu;
	//positive PASSED, negative FAILED
}
 //<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 //*****************************************************************************


/**
  * @brief  System Clock Configuration
  *         The system Clock is configured as follow :
  *            System Clock source            = PLL (HSE BYPASS)
  *            SYSCLK(Hz)                     = 280000000 (CPU Clock)
  *            HCLK(Hz)                       = 280000000 (AXI and AHBs Clock)
  *            AHB Prescaler                  = 1
  *            CD APB3 Prescaler              = 2 (APB3 Clock  140MHz)
  *            CD APB1 Prescaler              = 2 (APB1 Clock  140MHz)
  *            CD APB2 Prescaler              = 2 (APB2 Clock  140MHz)
  *            SRD APB4 Prescaler             = 2 (APB4 Clock  140MHz)
  *            HSE Frequency(Hz)              = 8000000
  *            PLL_M                          = 4
  *            PLL_N                          = 280
  *            PLL_P                          = 2
  *            PLL_Q                          = 4
  *            PLL_R                          = 2
  *            VDD(V)                         = 3.3
  *            Flash Latency(WS)              = 6
  * @param  None
  * @retval None
  */
static void SystemClock_Config(void)
{
  RCC_ClkInitTypeDef RCC_ClkInitStruct;
  RCC_OscInitTypeDef RCC_OscInitStruct;
  HAL_StatusTypeDef ret = HAL_OK;

  /*!< Supply configuration update enable */
  HAL_PWREx_ConfigSupply(PWR_DIRECT_SMPS_SUPPLY);

  /* The voltage scaling allows optimizing the power consumption when the device is
     clocked below the maximum system frequency, to update the voltage scaling value
     regarding system frequency refer to product datasheet.  */
  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

  while(!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {}

  /* Enable HSE Oscillator and activate PLL with HSE as source */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
  RCC_OscInitStruct.HSEState = RCC_HSE_BYPASS;
  RCC_OscInitStruct.HSIState = RCC_HSI_OFF;
  RCC_OscInitStruct.CSIState = RCC_CSI_OFF;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
  RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;

  RCC_OscInitStruct.PLL.PLLM = 4;
  RCC_OscInitStruct.PLL.PLLN = 280;
  RCC_OscInitStruct.PLL.PLLFRACN = 0;
  RCC_OscInitStruct.PLL.PLLP = 2;
  RCC_OscInitStruct.PLL.PLLR = 2;
  RCC_OscInitStruct.PLL.PLLQ = 4;

  RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1VCOWIDE;
  RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1VCIRANGE_1;
  RCC_OscInitStruct.PLL.PLLFRACN  = 0;

  ret = HAL_RCC_OscConfig(&RCC_OscInitStruct);
  if(ret != HAL_OK)
  {
    Error_Handler();
  }

/* Select PLL as system clock source and configure  bus clocks dividers */
  RCC_ClkInitStruct.ClockType = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_D1PCLK1 | RCC_CLOCKTYPE_PCLK1 | \
                                 RCC_CLOCKTYPE_PCLK2  | RCC_CLOCKTYPE_D3PCLK1);

  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
  RCC_ClkInitStruct.SYSCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_HCLK_DIV1;
  RCC_ClkInitStruct.APB3CLKDivider = RCC_APB3_DIV2;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_APB1_DIV2;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_APB2_DIV2;
  RCC_ClkInitStruct.APB4CLKDivider = RCC_APB4_DIV2;
  ret = HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_6);
  if(ret != HAL_OK)
  {
    Error_Handler();
  }

/*
  Note : The activation of the I/O Compensation Cell is recommended with communication  interfaces
          (GPIO, SPI, FMC, ...)  when  operating at  high frequencies(please refer to product datasheet)
          The I/O Compensation Cell activation  procedure requires :
        - The activation of the CSI clock
        - The activation of the SYSCFG clock
        - Enabling the I/O Compensation Cell : setting bit[0] of register SYSCFG_CCCSR

          To do this please uncomment the following code
*/

  /*
  __HAL_RCC_CSI_ENABLE() ;

  __HAL_RCC_SYSCFG_CLK_ENABLE() ;

  HAL_EnableCompensationCell();
  */

}


static void TFT_Display_Header(void)
{
	UTIL_LCD_Clear(UTIL_LCD_COLOR_WHITE);

	/* Set Menu font */
	UTIL_LCD_SetFont(&Font12);

	/* Set Text color */
	UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_RED);

	/* Display message */
	UTIL_LCD_DisplayStringAtLine(1, (uint8_t *)" NUCLEO-H7A3ZI-Q  ");
	UTIL_LCD_DisplayStringAtLine(2, (uint8_t *)" Benchmark        ");
}


static void TFT_Display_Desc(uint8_t *desc)
{
	UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_BLUE);
	/* Display message */
	UTIL_LCD_DisplayStringAtLine(4, (uint8_t *)desc);
	UTIL_LCD_DisplayStringAtLine(5, (uint8_t *)(desc + 18));
	UTIL_LCD_DisplayStringAtLine(6, (uint8_t *)(desc + 36));
}


static void TFT_Display_Menu(uint8_t *desc)
{
//	  JOYPin_TypeDef tmp;

	  /* Set Menu font */
	  UTIL_LCD_SetFont(&Font12);

	  /* Set Back color */
	  UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_WHITE);

	TFT_Display_Header();
	TFT_Display_Desc(desc);

	/* Set Text color */
	UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_WHITE);
	/* Display message */
	UTIL_LCD_DisplayStringAtLine(11, (uint8_t *)"  Press JOY DOWN  ");
	UTIL_LCD_DisplayStringAtLine(12, (uint8_t *)"  to continue...  ");


	/* Wait for JOY_DOWN is pressed */

	//while (BSP_JOY_GetState() != JOY_DOWN)
	while (ADAFRUIT_802_JOY_GetState(JOY1) != JOY_DOWN)
	{}
	HAL_Delay(200);


	TFT_Display_Header();
	TFT_Display_Desc(desc);
	UTIL_LCD_DisplayStringAtLine(11, (uint8_t *)"Running ...       ");
}

static void TFT_Display_result(uint8_t *desc, uint8_t *result, uint8_t *unit)
{
	TFT_Display_Header();
	TFT_Display_Desc(desc);

	UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_BLACK);
	/* Display message */
	UTIL_LCD_DisplayStringAtLine(8,  (uint8_t *)"Result");
	UTIL_LCD_DisplayStringAtLine(9,  (uint8_t *)result);
	UTIL_LCD_DisplayStringAtLine(10, (uint8_t *)unit);
}

static void TFT_Display_Progress(int current, uint32_t max)
{
	uint16_t pos;
	float scale;

	if (current == -1){
		UTIL_LCD_DrawRect(2, 100, 122, 10, UTIL_LCD_COLOR_BLACK);
	} else {
		if ( max > 120 ) {

			scale = 120.0f / (float)max;
			current = (int)((float)(current) * scale);
			pos = (uint16_t)(current + 1);

		} else {
			pos = (uint16_t)(((current + 1) * 120) / max);
		}
		UTIL_LCD_FillRect(3, 101, pos, 8, UTIL_LCD_COLOR_BLACK);
	}
}

/**
  * @brief  SD Card Configuration.
  * @param  None
  * @retval None
  */
static void SDCard_Config(void)
{
  uint32_t counter = 0;

  if(FATFS_LinkDriver(&SD_Driver, SD_Path) == 0)
  {
    /* Initialize the SD mounted on adafruit 1.8" TFT shield */
    if(ADAFRUIT_802_SD_Init(0) != BSP_ERROR_NONE)
    {
      TFT_DisplayErrorMessage(BSP_SD_INIT_FAILED);
    }

    /* Check the mounted device */
    if(f_mount(&SD_FatFs, (TCHAR const*)"/", 0) != FR_OK)
    {
      TFT_DisplayErrorMessage(FATFS_NOT_MOUNTED);
    }
    else
    {
      /* Initialize the Directory Files pointers (heap) */
      for (counter = 0; counter < MAX_BMP_FILES; counter++)
      {
        pDirectoryFiles[counter] = malloc(11);
      }
    }
  }
}



/**
  * @brief  Displays adequate message on TFT available on adafruit 1.8" TFT shield
  * @param  message: Error message to be displayed on the LCD.
  *   This parameter can be one of following values:
  *     @arg SD_CARD_NOT_FORMATTED: SD CARD is not FAT formatted
  *     @arg SD_CARD_FILE_NOT_SUPPORTED: File is not supported
  *     @arg SD_CARD_OPEN_FAIL: Failure to open directory
  *     @arg FATFS_NOT_MOUNTED: FatFs is not mounted
  * @retval None
  */
static void TFT_DisplayErrorMessage(uint8_t message)
{
  /* LCD Clear */
  UTIL_LCD_Clear(UTIL_LCD_COLOR_WHITE);
  /* Set Error Message Font */
  UTIL_LCD_SetFont(&Font12);
  /* Set Text and Back colors */
  UTIL_LCD_SetBackColor(UTIL_LCD_COLOR_GRAY);
  UTIL_LCD_SetTextColor(UTIL_LCD_COLOR_RED);

  if(message == SD_CARD_NOT_FORMATTED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAtLine(5, (uint8_t*)" SD Card is not    ");
    UTIL_LCD_DisplayStringAtLine(6, (uint8_t*)" FAT formatted.    ");
    UTIL_LCD_DisplayStringAtLine(7, (uint8_t*)" Please Format the ");
    UTIL_LCD_DisplayStringAtLine(8, (uint8_t*)" microSD card.     ");
    while (1)
    {
    }
  }
  if(message == SD_CARD_FILE_NOT_SUPPORTED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAtLine(5, (uint8_t*)"                   ");
    UTIL_LCD_DisplayStringAtLine(6, (uint8_t*)" File type is not  ");
    UTIL_LCD_DisplayStringAtLine(7, (uint8_t*)" supported.        ");
    UTIL_LCD_DisplayStringAtLine(8, (uint8_t*)"                   ");
    while(1)
    {
    }
  }
  if(message == SD_CARD_OPEN_FAIL)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAtLine(5, (uint8_t*)"                   ");
    UTIL_LCD_DisplayStringAtLine(6, (uint8_t*)" Open directory    ");
    UTIL_LCD_DisplayStringAtLine(7, (uint8_t*)" fails.            ");
    UTIL_LCD_DisplayStringAtLine(8, (uint8_t*)"                   ");
    while(1)
    {
    }
  }
  if(message == FATFS_NOT_MOUNTED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAtLine(5, (uint8_t*)"                   ");
    UTIL_LCD_DisplayStringAtLine(6, (uint8_t*)" Cannot mount      ");
    UTIL_LCD_DisplayStringAtLine(7, (uint8_t*)" FatFs on Drive.   ");
    UTIL_LCD_DisplayStringAtLine(8, (uint8_t*)"                   ");
    while (1)
    {
    }
  }
  if(message == BSP_SD_INIT_FAILED)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAtLine(5, (uint8_t*)"                   ");
    UTIL_LCD_DisplayStringAtLine(6, (uint8_t*)" SD Init           ");
    UTIL_LCD_DisplayStringAtLine(7, (uint8_t*)" fails.            ");
    UTIL_LCD_DisplayStringAtLine(8, (uint8_t*)"                   ");
    while(1)
    {
    }
  }
  if(message == SD_CARD_NO_FILE)
  {
    /* Display message */
    UTIL_LCD_DisplayStringAtLine(5, (uint8_t*)"                   ");
    UTIL_LCD_DisplayStringAtLine(6, (uint8_t*)" NO image found     ");
    UTIL_LCD_DisplayStringAtLine(7, (uint8_t*)" in the SD card.   ");
    UTIL_LCD_DisplayStringAtLine(8, (uint8_t*)"                   ");
    while(1)
    {
    }
  }
}

/**
  * @brief  Blinks LED2 with two frequencies depending on User press button.
  * @param  None
  * @retval None
  */
static void LED2_Blink(void)
{
  /* Configure LED2 on Nucleo */
  BSP_LED_Init(LED2);

  /* Configure the User Button in EXTI Mode */
  BSP_PB_Init(BUTTON_USER, BUTTON_MODE_EXTI);

  /* Initiate BlinkSpeed variable */
  BlinkSpeed = 0;

  /* Infinite loop */
  while(1)
  {
    /* Test on blink speed */
    if(BlinkSpeed == 0)
    {
      BSP_LED_Toggle(LED2);
      /* Wait for 500ms */
      HAL_Delay(500);
    }
    else if(BlinkSpeed == 1)
    {
      BSP_LED_Toggle(LED2);
      /* Wait for 100ms */
      HAL_Delay(100);
    }
    else if(BlinkSpeed == 2)
    {
      BSP_LED_Toggle(LED2);
      /* Wait for 50ms */
      HAL_Delay(50);
    }
  }
}

/**
  * @brief  BSP Push Button callback
  * @param  Button Specifies the pin connected EXTI line
  * @retval None
  */
void BSP_PB_Callback(Button_TypeDef Button)
{
  if(BlinkSpeed == 2)
  {
    BlinkSpeed = 0;
  }
  else
  {
    BlinkSpeed ++;
  }
}

/**
  * @brief  Check the availability of adafruit 1.8" TFT shield on top of STM32NUCLEO
  *         board. This is done by reading the state of IO PF.03 pin (mapped to
  *         JoyStick available on adafruit 1.8" TFT shield). If the state of PF.03
  *         is high then the adafruit 1.8" TFT shield is available.
  * @param  None
  * @retval SHIELD_DETECTED: 1.8" TFT shield is available
  *         SHIELD_NOT_DETECTED: 1.8" TFT shield is not available
  */
static ShieldStatus TFT_ShieldDetect(void)
{
  GPIO_InitTypeDef  GPIO_InitStruct;

  /* Enable GPIO clock */
  ADAFRUIT_802_ADCx_GPIO_CLK_ENABLE();

  GPIO_InitStruct.Pin = ADAFRUIT_802_ADCx_GPIO_PIN ;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_PULLDOWN;
  HAL_GPIO_Init(ADAFRUIT_802_ADCx_GPIO_PORT , &GPIO_InitStruct);

  if(HAL_GPIO_ReadPin(ADAFRUIT_802_ADCx_GPIO_PORT, ADAFRUIT_802_ADCx_GPIO_PIN) != 0)
  {
    return SHIELD_DETECTED;
  }
  else
  {
    return SHIELD_NOT_DETECTED;
  }
}

/**
  * @brief  This function is executed in case of error occurrence.
  * @param  None
  * @retval None
  */
static void Error_Handler(void)
{
  /* User may add here some code to deal with this error */
  while(1)
  {
  }
}

/**
  * @brief  CPU L1-Cache enable.
  * @param  None
  * @retval None
  */
static void CPU_CACHE_Enable(void)
{
  /* Enable I-Cache */
  SCB_EnableICache();

  /* Enable D-Cache */
  SCB_EnableDCache();
}

/**
  * @brief  Configure the MPU attributes
  * @param  None
  * @retval None
  */
static void MPU_Config(void)
{
  MPU_Region_InitTypeDef MPU_InitStruct;

  /* Disable the MPU */
  HAL_MPU_Disable();

  /* Configure the MPU as Strongly ordered for not defined regions */
  MPU_InitStruct.Enable = MPU_REGION_ENABLE;
  MPU_InitStruct.BaseAddress = 0x00;
  MPU_InitStruct.Size = MPU_REGION_SIZE_4GB;
  MPU_InitStruct.AccessPermission = MPU_REGION_NO_ACCESS;
  MPU_InitStruct.IsBufferable = MPU_ACCESS_NOT_BUFFERABLE;
  MPU_InitStruct.IsCacheable = MPU_ACCESS_NOT_CACHEABLE;
  MPU_InitStruct.IsShareable = MPU_ACCESS_SHAREABLE;
  MPU_InitStruct.Number = MPU_REGION_NUMBER0;
  MPU_InitStruct.TypeExtField = MPU_TEX_LEVEL0;
  MPU_InitStruct.SubRegionDisable = 0x87;
  MPU_InitStruct.DisableExec = MPU_INSTRUCTION_ACCESS_DISABLE;

  HAL_MPU_ConfigRegion(&MPU_InitStruct);

  /* Enable the MPU */
  HAL_MPU_Enable(MPU_PRIVILEGED_DEFAULT);
}


#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t* file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/**
  * @}
  */

/**
  * @}
  */

